/* ///////////////////////////////////////////////////////////////////////// */
/*  This is part of the source of the OMAP 5912 heterogeneous dual-core      */
/*  MPEG-4 SP video decoder published in ACM Transactions on Embedded        */
/*  Computing Systems, Vol. X, Issue Y.                                      */
/* ------------------------------------------------------------------------- */
/*  The source code is released under GPL license.                           */
/*                                                                           */
/*  Copyright, 2011                                                          */
/*  Multimedia Embedded Systems Labs                                         */
/*  Dept. of Computer Science                                                */
/*  National Chiao Tung University                                           */
/*  Hsinchu, Taiwan.                                                         */
/* ///////////////////////////////////////////////////////////////////////// */

/** ============================================================================
 *   @file  csl_icache.h
 *
 *   @path  $(CSLPATH)\dsp\icache\src
 *
 *   @desc  API header file for DSP ICACHE CSL
 *
 */

  /*  ===========================================================================
   *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004
   *
   *   Use of this software is controlled by the terms and conditions found in 
   *   the license agreement under which this software has been supplied.
   *   ==========================================================================
   */

/*  @(#) PSP/CSL 3.00.01.00[5912] (2004-05-01)  */

/* =============================================================================
 *  Revision History
 *  ===============
 *  08-Jul-2004 rr  Made modifications to support changes in CSL architecture
 *  01-May-2004 sks File Created.
 *
 * =============================================================================
 */

#ifndef _CSL_ICACHE_H_
#define _CSL_ICACHE_H_

#ifdef __cplusplus
extern  "C"
{
#endif

#include <csl.h>
#include <cslr_icache.h>

/**************************************************************************\
* DSP ICACHE macro declarations global within ICACHE CSL
\**************************************************************************/

/** Set CACLR field in the ST3_55 to 1) */
#define CSL_ICACHE_FLUSH                asm(" BSET CACLR")

/** Set CAEN field in the ST3_55 to 1) */
#define CSL_ICACHE_ENABLE               asm("   BSET CAEN")

/** Set CAEN field in the ST3_55 to 0) */
#define CSL_ICACHE_DISABLE              asm("   BCLR CAEN")

/** Set CAFRZ field in the ST3_55 to 1) */
#define CSL_ICACHE_FREEZE               asm("   BSET CAFRZ")

/** Set CAFRZ field in the ST3_55 to 0) */
#define CSL_ICACHE_UNFREEZE             asm("   BCLR CAFRZ")

/** Mask for CAFRZ field in DSP ST3 register */
#define CSL_ICACHE_ST3_CAFRZ_MASK        (0x8000)

/** Shift for CAFRZ field in DSP ST3 register */
#define CSL_ICACHE_ST3_CAFRZ_SHIFT       (15)

/** Mask for CAEN field in DSP ST3 register */
#define CSL_ICACHE_ST3_CAEN_MASK         (0x4000)

/** Shift for CAEN field in DSP ST3 register */
#define CSL_ICACHE_ST3_CAEN_SHIFT        (14)

/** Mask for CACLR field in DSP ST3 register */
#define CSL_ICACHE_ST3_CACLR_MASK        (0x2000)

/** Shift for CACLR field in DSP ST3 register */
#define CSL_ICACHE_ST3_CACLR_SHIFT        (13)

/**
 * DSP CPU ST3 register address
 */
#define CSL_ICACHE_DSP_ST3_REG       *((volatile Uint16*) 0x00000004)

/**************************************************************************\
* DSP ICACHE global typedef declarations
\**************************************************************************/

/**
 *  These enumeration values specify the various types of cache 
 *  flush operations supported by the DSP I-cache CSL. One of 
 *  these values is expected as last argument for the hardware 
 *  control command CSL_ICACHE_FLUSH. 
 */
    typedef enum
    {

    /** Flush all configured regions (whole cache). */
        CSL_ICACHE_FLUSH_REGION_ALL = 0,

    /** Flush a specific cache line. */
        CSL_ICACHE_FLUSH_REGION_LINE = 1,

    /** Flush N-way I-cache. */
        CSL_ICACHE_FLUSH_REGION_NWAY = 2,

    /** Flush first 1/2 ramset. */
        CSL_ICACHE_FLUSH_REGION_RAMSET_1 = 3,

    /** Flush second 1/2 ramset. */
        CSL_ICACHE_FLUSH_REGION_RAMSET_2 = 4,

    /** Flush both 1/2 ramset. */
        CSL_ICACHE_FLUSH_REGION_RAMSET_BOTH = 5
    } CSL_IcacheFlushRegion;

/**
 *  The enumeration specifies the various cache organizations, 
 *  which are possible in case of DSP I-cache. In case of DSP 
 *  I-cache setup operations; these enumeration values are used 
 *  to specify the cache organization in which to configure the 
 *  DSP I-cache. And in case of status query commands these values 
 *  are used to specify the cache organization in which the DSP 
 *  I-cache is enabled at the time of query.
 */
    typedef enum
    {

    /** Direct-mapped */
        CSL_ICACHE_1WAY_ORG = 0,

    /** Two-way set-associative */
        CSL_ICACHE_2WAY_ORG = 1,

    /** Direct-mapped and one 1/2 ramset */
        CSL_ICACHE_1WAY_1HFRAMSET_ORG = 2,

    /** Direct-mapped and two 1/2 ramset */
        CSL_ICACHE_1WAY_2HFRAMSET_ORG = 3,

    /** Two-way set-associative and one 1/2 ramset */
        CSL_ICACHE_2WAY_1HFRAMSET_ORG = 4,

    /** Two-way set-associative and two 1/2 ramset */
        CSL_ICACHE_2WAY_2HFRAMSET_ORG = 5,

    /** 
     *  Following enum value represents invalid cache configuration.
     */
        CSL_ICACHE_INVALID_CACHE_ORG = 6
    } CSL_IcacheOrgn;

/**
 *  This enumeration specifies the various possible values for 
 *  the configuration parameter '1/2 ramset number' present in 
 *  the register GCR. The enumeration is also used to reference 
 *  specific 1/2 ramset.
 */
    typedef enum
    {

    /** No 1/2 ramset */
        CSL_ICACHE_NOHFRAMSET = 0,

    /** One 1/2 ramset or first 1/2 ramset */
        CSL_ICACHE_1HFRAMSET = 1,

    /** Two 1/2 ramset or two 1/2 ramset*/
        CSL_ICACHE_2HFRAMSET = 2
    } CSL_IcacheHfRamset;

/**
 *  This enumeration specifies the various possible values for 
 *  the configuration parameter 'Way Number' present in the register 
 *  GCR. 
 */
    typedef enum
    {

    /** No N-Way */
        CSL_ICACHE_NOWAY = 0,

    /** 1-Way */
        CSL_ICACHE_1WAY = 1,

    /** 2-Way */
        CSL_ICACHE_2WAY = 2
    } CSL_IcacheNway;

/**
 *  This enum is used to convey the information whether to enable or 
 *  disable the specific configuration parameter.
 *  The enum is also used to return status for some of the status query 
 *  commands.
 */
    typedef enum
    {

    /** 
     *  For the hardware control commands, this enumeration value is
     *  used to specify the resetting the configuration parameter 
     *  associated with the control command.     
     *  For status query commands this enumeration value tells that 
     *  queried field is disabled.
     */
        CSL_ICACHE_CFG_PARAM_DISABLE = 0,

    /** 
     *  For the hardware control commands, this enumeration value is 
     *  used to specify the setting the configuration parameter 
     *  associated with the control command.     
     *  For status query commands this enumeration value tells that 
     *  queried field is enabled.
     */
        CSL_ICACHE_CFG_PARAM_ENABLE = 1
    } CSL_IcacheEnableDisable;

/**
 *  Status query commands that are supported by the DSP I-cache CSL.
 */
    typedef enum
    {

    /** 
     * @brief Gets the current configured organization of DSP I-cache. 
     * @param (CSL_IcacheEnableDisable *)
     */
        CSL_ICACHE_QUERY_GLOBAL_ENABLE = 0,

    /** 
     * @brief Gets the current value of the CAEN, CAFRZ and CACLR fields 
     *        in the ST3 register.
     * @param (CSL_IcacheST3Reg *) 
     */
        CSL_ICACHE_QUERY_ST3_REG = 1,

    /** 
     * @brief Gets the current value of I-cache_Enable field in the 
     *        I-cache Status Register.
     * @param (CSL_IcacheEnableDisable *)
     */
        CSL_ICACHE_QUERY_IS_ENABLED = 2,

    /** 
     * @brief Gets the current value of the Bus Error field in the I-cache 
     *        Status Register. 
     * @param (Uint16 *)
     */
        CSL_ICACHE_QUERY_BUS_ERROR = 3,

    /** 
     * @brief Gets the current value of the N-Way configuration parameters 
     *        present in the registers GCR and NWCR
     * @param (CSL_IcacheNwayCfg *) 
     */
        CSL_ICAHCE_QUERY_NWAY_CFG = 4,

    /** 
     * @brief Gets the current value of the 1/2 ramset configuration 
     *        parameters present in the register GCR. 
     * @param (CSL_IcacheGlblHfRamsetCfg *)
     */
        CSL_ICACHE_QUERY_GLBLHFRAMSET_CFG = 5,

    /** 
     * @brief Gets the current value of the 1/2 ramset configuration parameters 
     *        present in the registers RCR1/RCR2 and RTR1/RTR2. 
     * @param (CSL_IcacheHfRamsetCfg *)
     */
        CSL_ICACHE_QUERY_HFRAMSET_CFG = 6,

    /** 
     * @brief Gets the value of specific LVB bit in the LVB Status Registers. 
     * @param (CSL_IcacheLbvLruStatus  *)
     */
        CSL_ICACHE_QUERY_LVB_STATUS = 7,

    /** 
     * @brief Gets the value of specific LRU bit in the LRU Status Registers. 
     * @param (CSL_IcacheLbvLruStatus  *)
     */
        CSL_ICACHE_QUERY_LRU_STATUS = 8,

    /** 
     * @brief Gets the value of streaming enable bit in the GCR Register. 
     * @param (Bool *)
     */
        CSL_ICACHE_QUERY_STREAM_ENABLE = 9
    } CSL_IcacheHwStatusQuery;

/**
 * Enumeration for DSP ICACHE Control commands
 */
    typedef enum
    {

    /** 
     * @brief Command for enabling DSP I-cache 
     * @param None 
     */
        CSL_ICACHE_CMD_CACHE_ENABLE = 0,

    /** 
     * @brief Command for disabling DSP I-cache 
     * @param None
     */
        CSL_ICACHE_CMD_CACHE_DISABLE = 1,

    /** 
     * @brief The command sets the tag field in the RTR1 or RTR2 register 
     * @param (CSL_IcacheHfRamsetCfg *)
     */
        CSL_ICACHE_CMD_SET_HFRAMSET_TAG = 2,

    /** 
     * @brief Command for freezing and unfreezing the DSP I-cache 
     * @param (CSL_IcacheEnableDisable *)
     */
        CSL_ICACHE_CMD_SET_FREEZE_MODE = 3,

    /** 
     * @brief Command for flushing the DSP I-cache  
     * @param (CSL_IcacheFlushCmdArg *)
     */
        CSL_ICACHE_CMD_FLUSH_REGION_ALL = 4,

    /** 
     * @brief Command for flushing specific line of DSP I-cache  
     * @None
     */
        CSL_ICACHE_CMD_FLUSH_REGION_LINE = 5,

    /** 
     * @brief Command for flushing entire N-way cache region   
     * @param (CSL_IcacheFlushCmdArg *)
     */
        CSL_ICACHE_CMD_FLUSH_REGION_NWAY = 6,

    /** 
     * @brief Command for flushing entire half ramset 1   
     * @None
     */
        CSL_ICACHE_CMD_FLUSH_REGION_RAMSET_1 = 7,

    /** 
     * @brief Command for flushing entire half ramset 2   
     * @param (CSL_IcacheFlushCmdArg *)
     */
        CSL_ICACHE_CMD_FLUSH_REGION_RAMSET_2 = 8,

    /** 
     * @brief Command for flushing entire  ramset    
     * @param (CSL_IcacheFlushCmdArg *)
     */
        CSL_ICACHE_CMD_FLUSH_REGION_RAMSET_BOTH = 10,

    /** 
     * @brief The command sets or resets the Cut Clock field in the GCR 
     *        register 
     * @param (CSL_IcacheEnableDisable *)
     */
        CSL_ICACHE_CMD_SET_CUTCLOCK = 11
    } CSL_IcacheHwControlCmd;

/**
 * Enumeration for LVB Banks
 */
    typedef enum
    {

    /** Bank 0 for LVB bits. First Way */
        CSL_ICACHE_LVB_BANK0 = 0,

    /** Bank 1 for LVB bits. Second Way */
        CSL_ICACHE_LVB_BANK1 = 1,

    /** Bank 2 for LVB bits. Both half ramsets */
        CSL_ICACHE_LVB_BANK2 = 2
    } CSL_IcacheLVBBank;

/**
 * Enumeration for global enable configuration
 */
    typedef enum
    {

    /** Global enable configuration disabled */
        CSL_ICACHE_GLOBAL_ENABLE_DISABLE = 0,

    /** Global enable configuration enabled */
        CSL_ICACHE_GLOBAL_ENABLE_ENABLE = 1
    } CSL_IcacheGlobalEnable;

/**
 * Enumeration for global flush configuration
 */

    typedef enum
    {

    /** Global flush configuration disabled */
        CSL_ICACHE_GLOBAL_FLUSH_DISABLE = 0,

    /** Global flush configuration enabled */
        CSL_ICACHE_GLOBAL_FLUSH_ENABLE = 1
    } CSL_IcacheGlobalFlush;

/** 
 *  The default value for CSL_IcacheHwSetup structure 
 *  Cut Clock Mode - disabled
 *  Auto Gating    - enabled
 *  Cache Orgn     - 2Way set associative
 *  Global flush   - Disabled
 *  Global Enable  - Disabled
 */
#define CSL_ICACHE_HWSETUP_DEFAULTS {           \
            CSL_ICACHE_CFG_PARAM_ENABLE,        \
            CSL_ICACHE_CFG_PARAM_ENABLE,        \
            CSL_ICACHE_2WAY_ORG,                \
            CSL_ICACHE_GLOBAL_FLUSH_DISABLE,    \
            CSL_ICACHE_GLOBAL_ENABLE_DISABLE    \
            }

/**
 *  This structure is used for querying the bits of LVB and LRU 
 *  Status Registers.
 */
    typedef struct CSL_IcacheLbvLruStatus
    {

    /** Bank number. Valid values are 0 to 2. */
        CSL_IcacheLVBBank bankNum;

    /** Index number. Valid values are 0 to 511. */
        Uint16  index;

    /** Value of the bit corresponding to the bankNum and index. */
        Bool    currentVal;

    } CSL_IcacheLbvLruStatus;

/**
 *  This structure is used for querying CPU register ST3. Each 
 *  member of the structure represents a valid field of the register.
 */
    typedef struct CSL_IcacheST3Reg
    {

    /** Variable corresponding to CAFRZ field of CPU register ST3. */
        CSL_IcacheEnableDisable cacheFreeze;

    /** Variable corresponding to CAEN field of CPU register ST3. */
        CSL_IcacheEnableDisable cacheEnable;

    /** Variable corresponding to CACLR field of CPU register ST3. */
        CSL_IcacheEnableDisable cacheClear;

    } CSL_IcacheST3Reg;

/**
 *  This structure is used to pass parameters to the  
 *  CSL_IcacheHwControlCmd for the command CSL_ICACHE_CMD_FLUSH.
 */
    typedef struct CSL_IcacheFlushCmdArg
    {

    /** Region of the DSP I-cache which needs to be flushed. */
        CSL_IcacheFlushRegion flushRegion;

    /** 
     *  Argument is valid only when the value of the first member 
     *  flushRegion is CSL_ICACHE_FLUSH_REGION_LINE. The member 
     *  should be initialized with the virtual address corresponding 
     *  to which cache line is to be flushed.
     */
        Uint32  virtualAddr;

    } CSL_IcacheFlushCmdArg;

/**
 *  This structure is used to provide information related to 
 *  N-way I-cache configuration parameters.
 */
    typedef struct CSL_IcacheNwayCfg
    {

    /** Indicates presence of N-way. */
        Bool    nWayPresence;

    /** Indicates number of ways. */
        CSL_IcacheNway numWay;

    /** N-way specific DSP I-cache flush configuration. */
        CSL_IcacheEnableDisable nWayFlush;

    /** N-way specific DSP I-cache enable configuration. */
        CSL_IcacheEnableDisable nWayEnable;

    } CSL_IcacheNwayCfg;

/** 
 *  This structure is used to specify the 1/2 ramset configuration
 *  parameters to be written in the registers RCR1/RCR2 and RTR1/RTR2. 
 */
    typedef struct CSL_IcacheHfRamsetCfg
    {

    /** 
     *  1/2 ramset number. Possible range for this member is from 0 to 2. 
     *  Where 0 represents no 1/2 ramset, 1 represents first 1/2 ramset 
     *  and 2 represents second 1/2 ramset.t
     */
        CSL_IcacheHfRamset hfRamsetNumber;

    /** 
     *  1/2 ramset specific cache flush configuration. Based on the value 
     *  of first member hfRamsetNumber, the member refers to field 'flush' 
     *  in the register RCR1 or in the register RCR2
     */
        CSL_IcacheEnableDisable hfRamsetFlush;

    /**
     *  1/2 ramset specific cache enable configuration. Based on the value 
     *  of first member hfRamsetNumber, the member refers to the field 
     *  'Enable' in the register RCR1 or in the register RCR2.
     */
        CSL_IcacheEnableDisable hfRamsetEnable;

    /**
     *  1/2 ramset specific Tag valid field. Based on the value of first 
     *  member hfRamsetNumber, the member refers to field 'Tag valid' in 
     *  the register RCR1 or in the register RCR2.
     */
        Bool    hfRamsetTagValid;

    /**
     *  Tag for the 1/2 ramset. Based on the value of first member 
     *  hfRamsetNumber, the member refers to either register RTR1 or 
     *  register RTR2.
     */
        Uint16  hfRamsetTag;

    } CSL_IcacheHfRamsetCfg;

/**
 *  This structure is used to specify the 1/2 ramset configuration 
 *  parameters to be written in GCR register.
 */
    typedef struct CSL_IcacheGlblHfRamsetCfg
    {

    /** Indicates presence of 1/2 ramset */
        Bool    hfRamsetPresence;

    /** Indicates number of 1/2 ramset */
        CSL_IcacheHfRamset numHfRamset;

    } CSL_IcacheGlblHfRamsetCfg;

/**
 *  This structure used to setup DSP ICACHE
 */
    typedef struct CSL_IcacheHwSetup
    {

    /** Cut Clock Setting */
        CSL_IcacheEnableDisable disableClk;

    /** Autogating setting */
        CSL_IcacheEnableDisable autoGating;

    /** Cache organization */
        CSL_IcacheOrgn icacheOrgn;

    /** Global Flush configuration*/
        CSL_IcacheGlobalFlush icacheGlobalFlush;

    /** Global Enable configuration*/
        CSL_IcacheGlobalEnable icacheGlobalEnable;

    } CSL_IcacheHwSetup;

/**
 *   DSP ICACHE data object
 */
    typedef struct CSL_IcacheObj
    {

    /** icache instance */
        CSL_InstNum icacheNum;

    /** Pointer to the Global Register Overlay structure */
        CSL_IcacheRegsOvly regs;

    } CSL_IcacheObj;

/** 
 *   Pointer to the DSP ICACHE Object structure
 */
    typedef struct CSL_IcacheObj *CSL_IcacheHandle;

/** 
 * This will have the base-address information for the peripheral instance 
 */
    typedef struct
    {

    /** Base-address of the Configuration registers of the peripheral */
        CSL_IcacheRegsOvly regs;

    } CSL_IcacheBaseAddress;

/**
 * Icache Module specific parameters. Present implementation doesn't have
 * any module specific parameters.
 */
    typedef struct
    {

    /** Bit mask to be used for module specific parameters.
     *  The below declaration is just a place-holder for future
     *  implementation.
     */
        CSL_BitMask16 flags;

    } CSL_IcacheParam;

/** 
 * Icache module specific context information. Present implementation doesn't
 * have any Context information.
 */
    typedef struct
    {

    /** Context information of Icache.
     *  The below declaration is just a place-holder for future
     *  implementation.
     */
        Uint16  contextInfo;

    } CSL_IcacheContext;

/** 
 * Icache Config structure
 */
    typedef struct CSL_IcacheConfig
    {

        volatile Uint16 GCR;
        volatile Uint16 FLR0;
        volatile Uint16 FLR1;
        volatile Uint16 NWCR;
        volatile Uint16 RCR1;
        volatile Uint16 RTR1;
        volatile Uint16 RCR2;
        volatile Uint16 RTR2;

    } CSL_IcacheConfig;

/**
 * Default values for config structure
 */
#define CSL_ICACHE_CONFIG_DEFAULTS {   \
    CSL_ICACHE_GCR_RESETVAL,       \
    CSL_ICACHE_FLR0_RESETVAL,      \
    CSL_ICACHE_FLR1_RESETVAL,      \
    CSL_ICACHE_NWCR_RESETVAL,      \
    CSL_ICACHE_RCR1_RESETVAL,      \
    CSL_ICACHE_RTR1_RESETVAL,      \
    CSL_ICACHE_RCR2_RESETVAL,      \
    CSL_ICACHE_RTR2_RESETVAL       \
   }

/**************************************************************************\
*  global function declarations
\**************************************************************************/

/*
 * =============================================================================
 *   @func   CSL_icacheInit
 *   @desc
 *     This is DSP ICACHE module's init function. 
 *     Currently it does nothing. The function needs to be 
 *     called before using DSP ICACHE module.
 *
 *   @arg   pContext
 *          Context information for icache
 *
 *
 *   @ret  CSL_Status
 *         CSL_OK - Always returned
 *   @eg
 *      status = CSL_icacheInit(NULL);
 * =============================================================================
 */

    CSL_Status CSL_icacheInit(CSL_IcacheContext * pContext);

/*
 * ============================================================================
 *   @func   CSL_icacheOpen
 *
 *   @desc
 *     This function populates the peripheral data object for the instance and
 *     returns a handle to the instance
 *
 *   @arg   pIcacheObj
 *          Pointer to the icache instance object 
 *
 *   @arg   icacheNum
 *          Instance of the icache to be opened. 
 *   @arg   pIcacheParam
 *          Pointer to module specific parameters
 *
 *   @arg   pStatus
 *          pointer for returning status of the function call
 *    
 *   @ret  CSL_IcacheHandle
 *          Handle to the instance
 *  
 *   @eg
 *          CSL_status            status;
 *          CSL_IcacheObj          icacheObj;
 *          CSL_IcacheHandle       hIcache;
 *
 *          hIcache = CSL_icacheOpen (&icacheObj,          
 *                                      CSL_ICACHE, 
 *                                      NULL,  
 *                                      &status
 *                                      );
 *
 * =============================================================================
*/

    CSL_IcacheHandle CSL_icacheOpen(CSL_IcacheObj * icacheObj,
                                    CSL_InstNum icacheNum,
                                    CSL_IcacheParam * param,
                                    CSL_Status * status);

/*
 * =============================================================================
 *   @func CSL_icacheHwSetupRaw
 *
 *   @desc
 *      This function configures the registers of icache instance  as per the 
 *      values given in the Config structure.  
 *
 *   @arg   hIcache
 *          Handle to the icache instance
 *
 *   @arg   icacheConfig
 *          Pointer to the icache config structure
 *
 *   @ret   CSL_Status
 *          CSL_SOK - Configuration successful
 *          CSL_ESYS_BADHANDLE - Invalid handle
 *
 *   @eg
 *          CSL_status          status;
 *          CSL_IcacheConfig     icacheConfig;
 *          
 *          icacheConfig = CSL_ICACHE_CONFIG_DEFAULTS;
 *
 *          status = CSL_icacheHwsetupRaw (hIcache, &icacheConfig);
 *
 * =============================================================================
 */

    CSL_Status CSL_icacheHwSetupRaw(CSL_IcacheHandle hIcache,
                                    CSL_IcacheConfig * icacheConfig);

/*
 * ======================================================
 *   @func   CSL_icacheHwSetup
 *
 *   @desc
 *          This function sets the specified icache H/W parameters
 *
 *   @arg   hIcache
 *          The handle to the DSP icache instance
 *
 *   @arg   setup
 *          Pointer to the hardware setup structure
 *
 *   @ret   CSL_Status
 *          CSL_SOK - Successful, desired operation is done
 *          CSL_ESYS_INVPARAMS - Parameters passed is NULL
 *
 *   @eg
 *          CSL_IcacheHwSetup    setup;
 *          CSL_Status          status;
 *          ...
 *          status = CSL_icacheHwSetup(hIcache, &setup);
 *
 * =============================================================================
*/

    CSL_Status CSL_icacheHwSetup(CSL_IcacheHandle hIcache,
                                 CSL_IcacheHwSetup * hwSetup);

/*
 * =============================================================================
 *   @func   csl_icacheHwControl
 *
 *   @desc
 *          Function controls the icache peripheral specified by the hanldle.
 *          Operation done is based on the "cmd".
 *
 *   @arg   hIcache
 *          The handle to the DSP icache instance
 *
 *   @arg   cmd
 *          control command which needs to performed
 *
 *   @arg   cmdArg
 *          Argument related to the command
 *
 *   @ret   CSL_Status
 *          CSL_SOK - Successful, desired operation is done
 *          CSL_ESYS_INVCMD - Control command is not supported
 *
 *   @eg
 *      status = CSL_icacheHwControl(hIcache,CSL_ICACHE_CMD_CACHE_ENABLE,NULL);
 *
 * =============================================================================
*/

    CSL_Status CSL_icacheHwControl(CSL_IcacheHandle hIcache,
                                   CSL_IcacheHwControlCmd cmd, void *cmdArg);

/*
 * =============================================================================
 *   @func   CSL_icacheGetHwStatus
 *
 *   @desc
 *          This function is used to get the value of various parameters of an 
 *          instance of icache. The value returned depends on the query passed. 
 *
 *   @arg   hIcache
 *          Handle to icache object, if its NULL than CSL_ESYS_BADHANDLE
 *
 *   @arg   myQuery
 *          user's query
 *
 *   @arg   response
 *          Slot for saving response to query placed
 *
 *   @ret   CSL_Status
 *          CSL_SOK - Successful Retreive
 *          CSL_ESYS_INVQUERY - Query command is not supported
 *
 *   @eg
 *          status = CSL_icacheGetHwStatus (hIcache, 
 *                       CSL_ICACHE_QUERY_GLOBAL_ENABLE, 
 *                       (void*) &cfgVal);
 *
 * =============================================================================
*/

    CSL_Status CSL_icacheGetHwStatus(CSL_IcacheHandle hIcache,
                                     CSL_IcacheHwStatusQuery myQuery,
                                     void *response);

/*
 * =============================================================================
 *   @func   CSL_icacheClose
 *
 *   @desc
 *         This function closes the icache specified by its handle.
 *
 *   @arg   hIcache
 *          The handle to the DSP icache instance
 *
 *   @ret   CSL_Status
 *          CSL_SOK - Successful, desired operation is done
 *          CSL_ESYS_BADHANDLE - handle pass is invalid
 *
 *   @eg
 *     status = CSL_icacheClose(hIcache);
 *
 * =============================================================================
*/

    CSL_Status CSL_icacheClose(CSL_IcacheHandle hIcache);

/** ============================================================================
 *   @n@b CSL_icacheGetBaseAddress
 *
 *   @b Description
 *   @n  Function to get the base address of the peripheral instance.
 *       This function is used for getting the base address of the peripheral
 *       instance. This function will be called inside the CSL_icacheOpen()
 *       function call.This function is open for re-implementing if the user 
 *       wants to modify the base address of the peripheral object to point to
 *       a different location and there by allow CSL initiated write/reads into
 *       peripheral MMR's go to an alternate location. 
 *
 *   @b Arguments
 *   @verbatim      
            icacheNum        Specifies the instance of the Icache to be opened
 
            picacheParam     Module specific parameters
 
            pBaseAddress     Pointer to baseaddress structure containing base 
                             address details
            
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_OK            Open call is successful
 *   @li                    CSL_ESYS_FAIL     The instance number is invalid
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  Base Address structure is populated
 *
 *   @b Modifies
 *   @n  Base address structure is modified
 *
 *   @b Example
 *   @verbatim
        CSL_Status              status;
        CSL_IcacheBaseAddress   baseAddress;

        ...
        status = CSL_icacheGetBaseAddress(CSL_ICACHE, NULL, &baseAddress);

    @endverbatim
 * =============================================================================
 */

    extern CSL_Status CSL_icacheGetBaseAddress(CSL_InstNum icacheNum,
                                               CSL_IcacheParam * pIcacheParam,
                                               CSL_IcacheBaseAddress *
                                               pBaseAddress);

/*
 * =============================================================================
 *   @func   CSL_icacheGetHwSetup
 *
 *   @desc
 *          It retrives the hardware setup parameters of the I-cache module
 *          specified by the given handle. 
 *
 *   @arg   hIcache
 *          Handle to icache object, if its NULL than CSL_ESYS_BADHANDLE
 *
 *   @arg   hwSetup         
 *          Pointer to the harware setup structure
 *
 *
 *   @ret   CSL_Status
 *          CSL_SOK             - Retrieving the hardware                            
 *                                parameters is successfu                                                 
 *          CSL_ESYS_BADHANDLE  - The handle is passed is invalid
 *   @eg                          
 *          CSL_IcacheHandle   hIcache;
            CSL_IcacheHwSetup  hwSetup;

            ...
            
            status = CSL_icacheGetHwSetup(hIcache, &hwSetup);
 *          
 * =============================================================================
*/

    CSL_Status CSL_icacheGetHwSetup(CSL_IcacheHandle hIcache,
                                    CSL_IcacheHwSetup * hwSetup);

#ifdef __cplusplus
}
#endif

#endif                          /* _CSL_ICACHE_H_ */
